#-*- coding: utf-8 -*-

from pkg.logger import Logger
from dataclasses import dataclass
import time
import argparse
import json
import tomllib

class Config:
    def __init__(self, **kwargs):
        for name in kwargs:
            setattr(self, name, kwargs[name])

    def __eq__(self, other):
        if not isinstance(other, Namespace):
            return NotImplemented
        return vars(self) == vars(other)

    def __contains__(self, key):
        return key in self.__dict__
    
    def __str__(self):
        return json.dumps(self.__dict__)
    
@dataclass
class Runtime:
    __global_container__ = {}
    
    def __init__(self, config: Config = None, args: argparse.Namespace = None):
        self.config = config
        self.args = args
        self.__container__ = {}
    
    @staticmethod
    def set_global(key, value):
        Runtime.__global_container__[key] = value
        
    @staticmethod
    def get_global(key):
        return Runtime.__global_container__[key]
    
    @staticmethod
    def exists_global(key):
        return key in Runtime.__global_container__
    
    def set(self, key, value):
        self.__container__[key] = value
        
    def get(self, key):
        return self.__container__[key]   
    
    def exists(self, key):
        return key in self.__container__ 
        

class Bootstrap:
    def __init__(self):
        self.with_config = False
        self.start_at = time.time()
    
    def opt_with_config(self):
        self.with_config = True
        return self
    def parse_config(self, config_path: str):
        with open(config_path, 'rb') as f:
            config = tomllib.load(f)
            config_obj = Config()
            for key, value in config.items():
                setattr(config_obj, key, value)
        return config_obj
    
    def build_runtime(self, args_register_func):
        self.parser = argparse.ArgumentParser()
        if self.with_config:
            self.parser.add_argument('--conf', type=str, default='./conf/app.toml', required=True)
        if args_register_func:
            args_register_func(self.parser)
        args = self.parser.parse_args()
        
        if self.with_config:
            config = self.parse_config(args.conf)
        else:
            config = None
        
        return Runtime(config=config, args=args)

    def start(self, run_func, args_register_func=None):
        runtime = self.build_runtime(args_register_func)
        runtime.set_global('start_at', self.start_at)
        run_func(runtime)
        
def FacadeGet(key):
    if Runtime.exists_global(key):
        return Runtime.get_global(key)
    else:
        return None
def MLogger(module: str = '') -> Logger:
    key = f'logger.{module}'
    if not Runtime.exists_global(key):
        logger = Logger(module)
        Runtime.set_global(key, logger)
        return logger
    return FacadeGet(key)
    